from pyparsing import Literal, Optional, OneOrMore, Word, printables

from codeable_detectors.basic_detectors import AtLeastOneFileMatchesDetector
from codeable_detectors.detector_context import DetectorContext
from codeable_detectors.evidences import NamedEvidence, FailedEvidence
from codeable_detectors.pyparsing_patterns import round_braces_block, qualified_id


class JavaInvocation(AtLeastOneFileMatchesDetector):
    def __init__(self, invocation_name):
        super().__init__()
        self.file_endings = ["java"]
        self.invocation_name = invocation_name

    def detect_in_context(self, ctx, **kwargs):
        matches = ctx.matches_pattern(Literal(self.invocation_name) + round_braces_block + Literal(";"))
        if matches:
            return NamedEvidence(self.invocation_name, matches)
        return FailedEvidence("no invocation found")


def detect_java_import(ctx, package_name, import_name):
    # import [static] Identifier { . Identifier } [. *] ;
    # javaImportPattern = (Literal("import") + Optional(Literal("static")) + ID + 
    #     ZeroOrMore(Literal(".") + ID) + Optional(Literal(".") + Literal("*")) + Literal(";"))
    import_name_pattern = (Literal("*") | Literal(import_name))
    import_pattern = (Literal("import") + Optional(Literal("static")) +
                      Literal(package_name) + Literal(".") + import_name_pattern + Literal(";"))
    import_static_pattern = (Literal("import") + Literal("static") +
                             Literal(package_name) + Literal(".") + import_name_pattern + Literal(";"))

    matches = []
    import_matches = ctx.matches_pattern(import_pattern)
    for importMatch in import_matches:
        importMatch.update_keyword_args(import_name_pattern=import_name_pattern, package_name=package_name,
                                        isStaticImport=False)
        import_static_match = DetectorContext(importMatch).matches_pattern(import_static_pattern)
        if import_static_match:
            importMatch.update_keyword_args(isStaticImport=True)
        matches.append(importMatch)
    return matches


def get_java_variable_assignments(ctx):
    variable_assignment_pattern = qualified_id + Literal("=") + OneOrMore(Word(printables, excludeChars=";")) + Literal(
        ";")
    return ctx.matches_pattern(variable_assignment_pattern)
